/*
 * switch related alerts
 */
#include "libfma.h"
#include "lf_alert.h"
#include "lf_fabric.h"

#include "fms.h"
#include "fms_alert.h"
#include "fms_fabric.h"
#include "fms_error.h"
#include "fms_switch.h"
#include "fms_settings.h"
#include "fms_notify.h"

/*
 * Initialize alert structs in enclosures
 */
void
fms_fma_init_enclosure_alerts(
  struct lf_enclosure *ep)
{
  struct fms_alert *ap;

  /* allocate and initialize hosts alert anchor */
  LF_CALLOC(ap, struct fms_alert, 1);
  fms_init_alert_anchor(ap);

  FMS_ENC(ep)->alerts_anchor = ap;
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Initialize alert structs in a linecard
 */
void
fms_fma_init_linecard_alerts(
  struct lf_linecard *lp)
{
  struct fms_alert *ap;

  /* allocate and initialize linecard alert anchor */
  LF_CALLOC(ap, struct fms_alert, 1);
  fms_init_alert_anchor(ap);

  FMS_LC(lp)->alerts_anchor = ap;
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Initialize alert structs in an xbar
 */
void
fms_fma_init_xbar_alerts(
  struct lf_xbar *xp)
{
  struct fms_alert *ap;

  /* allocate and initialize xbar alert anchor */
  LF_CALLOC(ap, struct fms_alert, 1);
  fms_init_alert_anchor(ap);

  FMS_XBAR(xp)->alerts_anchor = ap;
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Initialize alert structs in a transceiver
 */
void
fms_fma_init_xcvr_alerts(
  struct lf_xcvr *xcp)
{
  struct fms_alert *ap;

  /* allocate and initialize xcvrs alert anchor */
  LF_CALLOC(ap, struct fms_alert, 1);
  fms_init_alert_anchor(ap);

  FMS_LC_XCVR(xcp)->alerts_anchor = ap;
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Initialize alert structs for a link
 */
void
fms_init_link_alerts(
  struct fms_link *linkp)
{
  struct fms_alert *ap;

  /* allocate and initialize hosts alert anchor */
  LF_CALLOC(ap, struct fms_alert, 1);
  fms_init_alert_anchor(ap);

  linkp->fl_alerts_anchor = ap;
  return;

 except:
  fms_perror_exit(1);
}

/*
 * An xbar port has been disabled
 */
void
fms_switch_alert_xbarport_disabled(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_XBARPORT_DISABLED);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_xbarport_disabled.enclosure,
      xp->linecard->enclosure->name);
  ap->alert.a.switch_xbarport_disabled.slot = lf_slot_display_no(xp->linecard);
  ap->alert.a.switch_xbarport_disabled.xbar = xp->xbar_no;
  ap->alert.a.switch_xbarport_disabled.port = port;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_XBAR(xp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * xbarport is no longer disabled
 */
void
fms_switch_alert_xbarport_enabled(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;
  struct fms_alert *nap;
  struct fms_alert *anchor;

  /*
   * Scan through all alerts on this xbar - if this port is marked as
   * disabled, relic the alert
   */
  anchor = FMS_XBAR(xp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    nap = ap->subj_next;	/* be safe */

    if (ap->alert.alert_type == LF_ALERT_SWITCH_XBARPORT_DISABLED
	&& ap->alert.a.switch_xbarport_disabled.port == port
	&& !ap->alert.relic) {
      fms_relic_alert(ap);
      return;
    }
    ap = nap;
  }
}

/*
 * An xbar port is down
 */
void
fms_switch_alert_xbarport_down(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_XBARPORT_DOWN);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_xbarport_down.enclosure,
      xp->linecard->enclosure->name);
  ap->alert.a.switch_xbarport_down.slot = lf_slot_display_no(xp->linecard);
  ap->alert.a.switch_xbarport_down.xbar = xp->xbar_no;
  ap->alert.a.switch_xbarport_down.port = port;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_XBAR(xp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * An xbar link is down
 */
void
fms_switch_alert_xbar_link_down(
  struct lf_xbar *xp,
  int port)
{
  union lf_node *pnp;

  /* If there is something on phys_ports[port], then this is an external
   * connection.  Othewise, it is an internal
   */
  pnp = xp->phys_ports[port];
  if (pnp == NULL || pnp->ln_type == LF_NODE_XBAR) {
    fms_switch_alert_xbar_int_link_down(xp, port);
  } else if (pnp->ln_type == LF_NODE_LC_XCVR) {
    struct lf_xcvr *xcp;
    union lf_node *onp;

    /* find what is on the other side of the xcvr */
    xcp = LF_XCVR(pnp);
    onp = xcp->ports[xp->phys_rports[port] - xcp->num_conns];

    /* link to another switch? */
    if (onp->ln_type == LF_NODE_LC_XCVR) {
      fms_switch_alert_xbar_ext_link_down(xp, port);

    /* link to a host? */
    } else if (onp->ln_type == LF_NODE_NIC_XCVR) {
      fms_switch_alert_xbar_host_link_down(xp, port);

    /* this is not true for XM XXX */
    } else {
      LF_ERROR(("Unsupported connection type"));
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * An xbar link is up
 */
void
fms_switch_alert_xbar_link_up(
  struct lf_xbar *xp,
  int port)
{
  struct fms_link *linkp;

  linkp = FMS_XBAR(xp)->x_port_data[port]->link;

  fms_switch_alert_link_up(linkp);
}

/*
 * A link is back up
 */
void
fms_switch_alert_link_up(
  struct fms_link *linkp)
{
  lf_alert_type_t types[3];

  /* relic all alerts indicating signal lost */
  types[0] = LF_ALERT_SWITCH_INT_LINK_DOWN;
  types[1] = LF_ALERT_SWITCH_EXT_LINK_DOWN;
  types[2] = LF_ALERT_HOST_LINK_DOWN;

  fms_alert_relic_matches_in_subj_list(linkp->fl_alerts_anchor, types, 3);
}

/*
 * A link internal to an enclosure is down
 */
void
fms_switch_alert_xbar_int_link_down(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;
  struct lf_xbar *oxp;
  int oport;
  struct switch_int_link_down_struct *adata;

  /* get other end of the connection */
  oxp = LF_XBAR(xp->topo_ports[port]);
  oport = xp->topo_rports[port];

  /* Must be an xbar */
  if (oxp->ln_type != LF_NODE_XBAR) {
    LF_ERROR(("Other end of phys conn must be xbar"));
  }

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_INT_LINK_DOWN);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.switch_int_link_down;
  LF_STRCPY(adata->enclosure, xp->linecard->enclosure->name);
  adata->slot1 = lf_slot_display_no(xp->linecard);
  adata->xbar1 = xp->xbar_no;
  adata->port1 = port;
  adata->slot2 = lf_slot_display_no(oxp->linecard);
  adata->xbar2 = oxp->xbar_no;
  adata->port2 = oport;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_XBAR(xp)->x_port_data[port]->link->fl_alerts_anchor,
      ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A link between enclosures is down
 */
void
fms_switch_alert_xbar_ext_link_down(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;
  struct lf_xcvr *xcp1;
  int xcport1;
  struct lf_xcvr *xcp2;
  int xcport2;
  struct switch_ext_link_down_struct *adata;

  /* get xcvr for this end */
  xcp1 = LF_XCVR(xp->phys_ports[port]);
  xcport1 = xp->phys_rports[port];

  /* get the other end */
  xcp2 = LF_XCVR(xcp1->ports[xcport1 - xcp1->num_conns]);
  xcport2 = xcp1->rports[xcport1 - xcp1->num_conns];

  /* make sure the other end is what we expect */
  if (xcp2->ln_type != LF_NODE_LC_XCVR) {
    LF_ERROR(("Other end of phys conn must be linecard xcvr"));
  }

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_EXT_LINK_DOWN);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.switch_ext_link_down;
  LF_STRCPY(adata->enclosure1, xcp1->p.linecard->enclosure->name);
  adata->slot1 = lf_slot_display_no(xcp1->p.linecard);
  adata->port1 = xcp1->p.linecard->xcvr_labels[xcp1->port];
  adata->subport1 = xcport1 - xcp1->num_conns;

  LF_STRCPY(adata->enclosure2, xcp2->p.linecard->enclosure->name);
  adata->slot2 = lf_slot_display_no(xcp2->p.linecard);
  adata->port2 = xcp2->p.linecard->xcvr_labels[xcp2->port];
  adata->subport2 = xcport2;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_XBAR(xp)->x_port_data[port]->link->fl_alerts_anchor,
      ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A link between an enclosure and a host is down
 */
void
fms_switch_alert_xbar_host_link_down(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;
  struct lf_xcvr *xcp1;
  int xcport1;
  struct lf_xcvr *xcp2;
  int xcport2;
  struct host_link_down_struct *adata;

  /* get xcvr for this end */
  xcp1 = LF_XCVR(xp->phys_ports[port]);
  xcport1 = xp->phys_rports[port];

  /* get the other end */
  xcp2 = LF_XCVR(xcp1->ports[xcport1 - xcp1->num_conns]);
  xcport2 = xcp1->rports[xcport1 - xcp1->num_conns];

  /* make sure the other end is what we expect */
  if (xcp2->ln_type != LF_NODE_NIC_XCVR) {
    LF_ERROR(("Other end of phys conn must be NIC xcvr"));
  }

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_HOST_LINK_DOWN);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.host_link_down;
  LF_STRCPY(adata->enclosure, xcp1->p.linecard->enclosure->name);
  adata->slot = lf_slot_display_no(xcp1->p.linecard);
  adata->port = xcp1->p.linecard->xcvr_labels[xcp1->port];
  adata->subport = xcport1 - xcp1->num_conns;

  LF_STRCPY(adata->hostname, xcp2->p.nic->host->hostname);
  adata->nic = xcp2->p.nic->host_nic_id;
  adata->nic_interface = xcp2->rports[xcport2 + xcp2->num_conns];

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_XBAR(xp)->x_port_data[port]->link->fl_alerts_anchor,
      ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * xbarport is no longer down
 */
void
fms_switch_alert_xbarport_up(
  struct lf_xbar *xp,
  int port)
{
  struct fms_alert *ap;
  struct fms_alert *nap;
  struct fms_alert *anchor;

  /*
   * Scan through all alerts on this xbar - if this port is marked as
   * down, relic the alert
   */
  anchor = FMS_XBAR(xp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    nap = ap->subj_next;	/* be safe */

    if (ap->alert.alert_type == LF_ALERT_SWITCH_XBARPORT_DOWN
	&& ap->alert.a.switch_xbarport_disabled.port == port
	&& !ap->alert.relic) {
      fms_relic_alert(ap);
      return;
    }
    ap = nap;
  }
}

/*
 * A transceiver has been disabled
 */
void
fms_switch_alert_xcvr_disabled(
  struct lf_xcvr *xcp)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_XCVR_DISABLED);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_xcvr_disabled.enclosure,
      xcp->p.linecard->enclosure->name);
  ap->alert.a.switch_xcvr_disabled.slot =
      lf_slot_display_no(xcp->p.linecard);
  ap->alert.a.switch_xcvr_disabled.port =
      xcp->p.linecard->xcvr_labels[xcp->port];

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_LC_XCVR(xcp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Transceiver is no longer disabled
 */
void
fms_switch_alert_xcvr_enabled(
  struct lf_xcvr *xcp)
{
  lf_alert_type_t types[1];

  /* relic all alerts of indicating xcvr disabled */
  types[0] = LF_ALERT_SWITCH_XCVR_DISABLED;

  fms_alert_relic_matches_in_subj_list(FMS_LC_XCVR(xcp)->alerts_anchor,
                                       types, 1);
}

/*
 * A transceiver has lost signal
 */
void
fms_switch_alert_xcvr_signal_lost(
  struct lf_xcvr *xcp)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_XCVR_SIGNAL_LOST);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_xcvr_signal_lost.enclosure,
      xcp->p.linecard->enclosure->name);
  ap->alert.a.switch_xcvr_signal_lost.slot =
      lf_slot_display_no(xcp->p.linecard);
  ap->alert.a.switch_xcvr_signal_lost.port =
      xcp->p.linecard->xcvr_labels[xcp->port];

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_LC_XCVR(xcp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Transceiver signal no longer lost
 */
void
fms_switch_alert_xcvr_signal_ok(
  struct lf_xcvr *xcp)
{
  lf_alert_type_t types[1];

  /* relic all alerts indicating signal lost */
  types[0] = LF_ALERT_SWITCH_XCVR_SIGNAL_LOST;

  fms_alert_relic_matches_in_subj_list(FMS_LC_XCVR(xcp)->alerts_anchor,
                                       types, 1);
}

/*
 * A linecard has experienced overtemps
 */
void
fms_switch_alert_linecard_overtemp(
  struct lf_linecard *lp)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_LINECARD_OVERTEMP);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_linecard_overtemp.enclosure,
            lp->enclosure->name);
  ap->alert.a.switch_linecard_overtemp.slot = lf_slot_display_no(lp);

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_LC(lp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A linecard is running hot
 */
void
fms_switch_alert_linecard_hot(
  struct lf_linecard *lp)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_LINECARD_HOT);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_linecard_hot.enclosure, lp->enclosure->name);
  ap->alert.a.switch_linecard_hot.slot = lf_slot_display_no(lp);

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_LC(lp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A hot linecard has cooled
 */
void
fms_switch_alert_linecard_cool(
  struct lf_linecard *lp)
{
  lf_alert_type_t types[1];

  /* relic all alerts indicating signal lost */
  types[0] = LF_ALERT_SWITCH_LINECARD_HOT;

  fms_alert_relic_matches_in_subj_list(FMS_LC(lp)->alerts_anchor, types, 1);
}

/*
 * Cannot contact monitoring linecard on an enclosure
 */
void
fms_switch_alert_cannot_read_enclosure(
  struct lf_enclosure *ep)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_SWITCH_CANNOT_READ);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  LF_STRCPY(ap->alert.a.switch_cannot_read.enclosure, ep->name);

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_ENC(ep)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * re-established contact with monitoring line card
 */
void
fms_switch_alert_enclosure_read_ok(
  struct lf_enclosure *ep)
{
  lf_alert_type_t types[1];

  /* relic all alerts indicating signal lost */
  types[0] = LF_ALERT_SWITCH_CANNOT_READ;

  fms_alert_relic_matches_in_subj_list(FMS_ENC(ep)->alerts_anchor, types, 1);
}

/*
 * A port is flip-flopping up and down too much
 */
void
fms_switch_alert_updown_count(
  struct lf_xbar *xp,
  int port)
{
  struct fms_settings *fsp;
  struct fms_alert *ap;
  struct fms_alert *anchor;
  struct switch_xbarport_updown_count_struct *adata;
  int this_alert;

  fsp = F.settings;

  /*
   * First, do the ugly work of looking for an alert of this
   * type in the xbar's list of alerts.  If we find an active alert already
   * on this port, just return.
   */
  this_alert = LF_ALERT_SWITCH_XBARPORT_UPDOWN_COUNT;
  anchor = FMS_XBAR(xp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert
	&& ap->alert.a.switch_xbarport_updown_count.port == port) {
      return;
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.switch_xbarport_updown_count;
  LF_STRCPY(adata->enclosure, xp->linecard->enclosure->name);
  adata->slot = lf_slot_display_no(xp->linecard);
  adata->xbar = xp->xbar_no;
  adata->port = port;
  adata->updown_count = FMS_XBAR(xp)->x_port_data[port]->vlfmon.portflipcount;
  adata->seconds = fsp->very_low_freq_monitor_interval;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * An xbar link has badcrc count
 */
void
fms_switch_alert_badcrc_count(
  struct lf_xbar *xp,
  int port,
  int fatal)
{
  union lf_node *pnp;

  /* If there is something on phys_ports[port], then this is an external
   * connection.  Othewise, it is an internal
   */
  pnp = xp->phys_ports[port];
  if (pnp == NULL || pnp->ln_type == LF_NODE_XBAR) {
    fms_switch_alert_xbar_int_link_badcrc(xp, port, fatal);
  } else if (pnp->ln_type == LF_NODE_LC_XCVR) {
    struct lf_xcvr *xcp;
    union lf_node *onp;

    /* find what is on the other side of the xcvr */
    xcp = LF_XCVR(pnp);
    onp = xcp->ports[xp->phys_rports[port] - xcp->num_conns];

    /* If disconnected, just ignore it */
    if (onp == NULL) return;

    /* link to another switch? */
    if (onp->ln_type == LF_NODE_LC_XCVR) {
      fms_switch_alert_xbar_ext_link_badcrc(xp, port, fatal);

    /* link to a host? */
    } else if (onp->ln_type == LF_NODE_NIC_XCVR) {
      fms_switch_alert_xbar_host_link_badcrc(xp, port);

    /* this is not true for XM XXX */
    } else {
      LF_ERROR(("Unsupported connection type"));
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * A link internal to an enclosure has excessive badcrc count
 */
void
fms_switch_alert_xbar_int_link_badcrc(
  struct lf_xbar *xp,
  int port,
  int fatal)
{
  struct fms_settings *fsp;
  struct fms_alert *ap;
  struct lf_xbar *oxp;
  int oport;
  int this_alert;
  struct fms_alert *anchor;
  struct switch_int_link_badcrc_count_struct *adata;

  fsp = F.settings;

  /* get other end of the connection */
  oxp = LF_XBAR(xp->topo_ports[port]);
  oport = xp->topo_rports[port];

  /* This can happen on XM cards */
  if (oxp == NULL) {
    fms_notify(FMS_EVENT_INFO, "badcrcs on XM-style card?");
    return;	/* XXX  deal with XM/2Z later */
  }

  /* Must be an xbar */
  if (oxp->ln_type != LF_NODE_XBAR) {
    LF_ERROR(("Other end of phys conn must be xbar"));
  }

  /*
   * First, do the ugly work of looking for an alert of this type in the
   * link's list of alerts.  If we find an active alert already,
   * just return.
   */
  this_alert = LF_ALERT_SWITCH_INT_LINK_BADCRC_COUNT;
  anchor = FMS_XBAR(xp)->x_port_data[port]->link->fl_alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert) {
      return;
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.switch_int_link_badcrc_count;
  LF_STRCPY(adata->enclosure, xp->linecard->enclosure->name);
  adata->slot1 = lf_slot_display_no(xp->linecard);
  adata->xbar1 = xp->xbar_no;
  adata->port1 = port;
  adata->slot2 = lf_slot_display_no(oxp->linecard);
  adata->xbar2 = oxp->xbar_no;
  adata->port2 = oport;
  adata->badcrc_count = FMS_XBAR(xp)->x_port_data[port]->lfmon.badcrcs;
  adata->seconds = fsp->low_freq_monitor_interval;
  if (fatal) {
    strcpy(adata->extra_text, " - excessive, port disabled");
  } else {
    strcpy(adata->extra_text, "");
  }

  /* register the alert */
  fms_register_alert(ap);

  /* link it into xbar list */
  fms_subj_link_alert(anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A link between enclosures has a high badcrc count
 */
void
fms_switch_alert_xbar_ext_link_badcrc(
  struct lf_xbar *xp,
  int port,
  int fatal)
{
  struct fms_settings *fsp;
  struct fms_alert *ap;
  struct lf_xcvr *xcp1;
  int xcport1;
  struct lf_xcvr *xcp2;
  int xcport2;
  int this_alert;
  struct fms_alert *anchor;
  struct switch_ext_link_badcrc_count_struct *adata;

  fsp = F.settings;

  /* get xcvr for this end */
  xcp1 = LF_XCVR(xp->phys_ports[port]);
  xcport1 = xp->phys_rports[port];

  /* get the other end */
  xcp2 = LF_XCVR(xcp1->ports[xcport1 - xcp1->num_conns]);
  xcport2 = xcp1->rports[xcport1 - xcp1->num_conns];

  /* make sure the other end is what we expect */
  if (xcp2->ln_type != LF_NODE_LC_XCVR) {
    LF_ERROR(("Other end of phys conn must be linecard xcvr"));
  }

  /*
   * First, do the ugly work of looking for an alert of this type in the
   * link's list of alerts.  If we find an active alert already,
   * just return.
   */
  this_alert = LF_ALERT_SWITCH_EXT_LINK_BADCRC_COUNT;
  anchor = FMS_XBAR(xp)->x_port_data[port]->link->fl_alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert) {
      return;
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.switch_ext_link_badcrc_count;
  LF_STRCPY(adata->enclosure1, xcp1->p.linecard->enclosure->name);
  adata->slot1 = lf_slot_display_no(xcp1->p.linecard);
  adata->port1 = xcp1->p.linecard->xcvr_labels[xcp1->port];
  adata->subport1 = xcport1 - xcp1->num_conns;

  LF_STRCPY(adata->enclosure2, xcp2->p.linecard->enclosure->name);
  adata->slot2 = lf_slot_display_no(xcp2->p.linecard);
  adata->port2 = xcp2->p.linecard->xcvr_labels[xcp2->port];
  adata->subport2 = xcport2;

  adata->badcrc_count = FMS_XBAR(xp)->x_port_data[port]->lfmon.badcrcs;
  adata->seconds = fsp->low_freq_monitor_interval;
  if (fatal) {
    strcpy(adata->extra_text, " - excessive, port disabled");
  } else {
    strcpy(adata->extra_text, "");
  }

  /* register the alert */
  fms_register_alert(ap);

  /* link it into xbar list */
  fms_subj_link_alert(anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A link between an enclosure and a host has a high badcrc count from
 * the xbar side
 */
void
fms_switch_alert_xbar_host_link_badcrc(
  struct lf_xbar *xp,
  int port)
{
  struct fms_settings *fsp;
  struct fms_alert *ap;
  struct lf_xcvr *xcp1;
  int xcport1;
  struct lf_xcvr *xcp2;
  int xcport2;
  int this_alert;
  struct fms_alert *anchor;
  struct switch_host_link_badcrc_count_struct *adata;

  fsp = F.settings;

  /* get xcvr for this end */
  xcp1 = LF_XCVR(xp->phys_ports[port]);
  xcport1 = xp->phys_rports[port];

  /* get the other end */
  xcp2 = LF_XCVR(xcp1->ports[xcport1 - xcp1->num_conns]);
  xcport2 = xcp1->rports[xcport1 - xcp1->num_conns];

  /* make sure the other end is what we expect */
  if (xcp2->ln_type != LF_NODE_NIC_XCVR) {
    LF_ERROR(("Other end of phys conn must be NIC xcvr"));
  }

  /*
   * First, do the ugly work of looking for an alert of this type in the
   * link's list of alerts.  If we find an active alert already,
   * just return.
   */
  this_alert = LF_ALERT_SWITCH_HOST_LINK_BADCRC_COUNT;
  anchor = FMS_XBAR(xp)->x_port_data[port]->link->fl_alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert) {
      return;
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.switch_host_link_badcrc_count;
  LF_STRCPY(adata->enclosure, xcp1->p.linecard->enclosure->name);
  adata->slot = lf_slot_display_no(xcp1->p.linecard);
  adata->port = xcp1->p.linecard->xcvr_labels[xcp1->port];
  adata->subport = xcport1 - xcp1->num_conns;

  LF_STRCPY(adata->hostname, xcp2->p.nic->host->hostname);
  adata->nic = xcp2->p.nic->host_nic_id;
  adata->nic_interface = xcp2->rports[xcport2 + xcp2->num_conns];

  adata->badcrc_count = FMS_XBAR(xp)->x_port_data[port]->lfmon.badcrcs;
  adata->seconds = fsp->low_freq_monitor_interval;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into xbar list */
  fms_subj_link_alert(anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}
